/**
 * \file           ir_receiver.cpp
 * \brief          Ir RC5 Decoding & Parser/Process Tasks
 * \author         Dennis Hromin
 * \date           12/18/2008
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "ir_receiver.h"
#include "os.h"
#include "Cresnet.h"

IrReceiver * g_IrReceiver;

/**
 * \author    Dennis Hromin
 * \brief     Callback function for Ir packet parser main task;  This is a C wrapper
 *            to access the non-static method of the Ir Class; once called,
 *            the method never returns
 * \date      12/18/2008
 * \param     param - anything passed in is ignored by this func
 * \return    void
 * \retval    none
**/
void IrRC5ProcessTaskCallback(UINT32 param)
{
        g_IrReceiver->RC5_IrProcessTask();
}

/**
 * \author        Dennis Hromin
 * \brief         constructor for IrReceiver class
 * \date          12/18/2008
 * \param         void
 * \return        void
 * \retval        none
**/
IrReceiver::IrReceiver(void):
    m_CodeAvailable(0),
	m_Dock(0),
	m_IrRC5Task(0),
	m_IrRC5Queue(0)
{
    UINT16 sStackSize;
    char TaskName[4];

// determine the stack size; use the greater of the minimum for the processor
    sStackSize = OsGetDefaultStackVarCnt();
    if ( sStackSize < IR_RC5_TASK_STACK )
        sStackSize = IR_RC5_TASK_STACK;

// create message queue for IR Received captures & main IR Receiver Task
    m_IrRC5Queue = OsQueueCreate(IR_RC5_QUEUE_LENGTH,sizeof(UINT16) );

// this is done using a C wrapper funtion for the class since a callback function
// is required and its difficult to do callbacks in C++ without templates
    sprintf((char *)&TaskName[0], "IR1");
    m_IrRC5Task = OsCreateNamedAdvTask(IrRC5ProcessTaskCallback, sStackSize, (UINT32 *)g_IrReceiver,
                                       IR_RC5_TASK_PRIORITY,  (const signed char *)(&TaskName[0]) );

    memset((void*)&m_RC5, 0, sizeof(m_RC5));
    memset((void*)&m_RF1way, 0, sizeof(m_RF1way));
}

/*------------------------------------------------------------------------------------------------*
For fixed format Cresnet II RF, the bit format is:
Type 09h: 1-Way RF Receiver

<id> <cnt> <09> <data2> <data1> <data0>

For 1-Way receivers reporting back to the Cresnet Master (ID 02), a typical packet is:
<02> <04> <09> <data2> <data1> <data0>

In response to a poll, data field is zero if no signal present, else it is framed 24b data.

For fixed format Cresnet II RF, the bit format is:

 011110xx x0xxx0xx x0xxx0xx
       ii i iii ii k kkk kk
       76 5 432 10 5 432 10

Where iiiiiiii is the 8-bit RF ID code, and kkkkkk is a 6 bit # (join 1-64) of the button #.
*-------------------------------------------------------------------------------------------------*/

/**
 * \author        Dennis Hromin
 * \brief         Main Ir Processing task that waits on Ir RC5 Decoded Queue for next packet code to be parsed
 * \date          12/24/2008
 * \param         void
 * \return        void
 * \retval        none
**/
void IrReceiver::RC5_IrProcessTask(void)
{
    UINT16 RC5data;  //temp pointer to the dequeued message
    UINT32 join;    //Join number to send back (Default is zero)
    UINT8 rc5_addr, rc5_cmd;
    BOOL newcode = FALSE;  //flag to indiciate whether new code was dequeued or not
    m_RC5.IRCodeState = RC5_CODE_STATE_RELEASE; //always startup in default release code state

    UINT32 current_time_msec = 0;
    UINT32 prev_time_msec = 0;

    while (1)
    {
        /* Wait at most 10 ms (10 ticks) for IRcode,  then continue to process the code if one exists and/or decrement timeout counter */
        m_CodeAvailable = OsDequeueWait(m_IrRC5Queue, (void *)&RC5data, RC5_CODE_TIMEOUT_INTERVAL_DELAY, FALSE);

        /* If there was a successful dequeue, process the RC5 data code */
        if (m_CodeAvailable >= 0)    //If m_CodeAvailable = -1, then nothing was returned
        {
            m_RC5.Code = (UINT16)RC5data; //Stores the dequeued RC5 code

            // 1st, For IR Docking Application, force timeout to zero, don't execute other code for NULL IR COde type.
            if(m_Dock != 0)
            {
                m_RC5.TimeOut = 0; //force timeout to always be zero for IRDock Applications
                if (m_RC5.Code != m_RC5.PrevCode)
                {
                    // A new code has been received, execute the release action of the previous code
                    newcode = TRUE;
                }
            }
            /* 2nd for normal RC5 Apps, see if there was a special "NULL" code enqueued... this means a packet was dropped mistream, so restart timeout timer */
            else if (m_RC5.Code == RC5_CODE_NULL)
            {
                newcode = FALSE; //invalid RC5 code, so set state to FALSE
                m_RC5.TimeOut = RC5_CODE_TIMEOUT; //restart the Timeout timer and continue forward and do nothing
            }

            /* Second, check if repeat code based on fact that timeout timer did not expire yet */
            else if (m_RC5.TimeOut)
            {
                if (m_RC5.Code != m_RC5.PrevCode)
                {
                    // A new code has been received, execute the release action of the previous code
                    newcode = TRUE;
                }
                else
                {
                    /*-- Allow IR code to be repeatedly processed (once every 480 milliseconds) if a button on remote is being held down --*/
                    current_time_msec = HwGetMsec();

                    if ( current_time_msec - prev_time_msec >= 480 )
                    {
                        newcode = TRUE;
                    }
                    else
                    {
                        newcode = FALSE;
                    }
                }

                // Retrigger the code timeout
                m_RC5.TimeOut = RC5_CODE_TIMEOUT;
            }

            /*-- Timeout has already happened and NOT counting.. so flag as new code and send out--*/
            else
            {
                // New code, execute the new code
                newcode = TRUE;
            }

            /*-- We determined there is a new code, so send out to master via network --*/
            if (newcode)
            {
                rc5_addr = (( m_RC5.Code >> 6) & 0x1F);
                rc5_cmd = (m_RC5.Code & 0x3F);

                m_RF1way.data0 = 0x78;
                m_RF1way.data1 = (UINT8)(((rc5_addr << 1) & 0x38) | (rc5_addr & 0x03));
                m_RF1way.data2 = (UINT8)((rc5_cmd & 0x03) | ((rc5_cmd << 1) & 0x38) | ((rc5_cmd << 2) & 0x80) );

                join = 0; //Join number forced to zero
                CresnetSlaveSendRF1Way(join, (UINT8)m_RF1way.data0, (UINT8)m_RF1way.data1, (UINT8)m_RF1way.data2 );

                // Save the new code and its timeout
                m_RC5.PrevCode = m_RC5.Code;
                if(m_Dock != 0)
                {
                   m_RC5.TimeOut = 0; //force timeout to always be zero for IRDock Applications
	     		}
                else
                {
                   m_RC5.TimeOut = RC5_CODE_TIMEOUT; //reset the timeout value
                }
                prev_time_msec = HwGetMsec();

                newcode = FALSE; //reset to default setting
            }
        }


        /* We Must decrement timeout timer to timeout any exisitng valid code, if timeout occurred, do nothing */
        if (m_RC5.TimeOut >= RC5_CODE_TIMEOUT_INTERVAL_DELAY)
        {
            m_RC5.TimeOut = m_RC5.TimeOut - RC5_CODE_TIMEOUT_INTERVAL_DELAY; //take off 10 msec
            if(m_RC5.TimeOut < RC5_CODE_TIMEOUT_INTERVAL_DELAY )  //if we hit less than min '10ms' and timed out
            {
                //      Execute the key release code
                join = 0; //Join number forced to zero
                //Sends out all 0's for the data bytes to indicate that button pressed was released //
                CresnetSlaveSendRF1Way(join, 0x00, 0x00, 0x00 );
            }
        }
            /* Else otherwise it must be busy, so loop to top and try again later */
   }// end while(1) loop
}//end task func



/**
 * \author        Dennis Hromin
 * \brief         Function to initialize/restart RC5 IR Capture to begin looking at new packet/ pulse samples
 * \date          12/26/2008
 * \param         void
 * \return        void
 * \retval        none
**/
void IrReceiver::RC5_RestartCapture(void)
{
        //Reset these two key values to default startup state
        m_RC5.NextSampleEdge    = 0;
        m_RC5.SampleHalfBitsCnt = 0;

        /*Disable TIMx Input Capture interrupt*/
        /*Disable TIMx Output Compare interrupt*/
        RC5_DisableInputCaptureIRQ();
        RC5_DisableInputTimeoutIRQ();

        //      Set for falling edge interrupt
        RC5_SetInputCaptEdge_Falling();

        /* clear All Input Capture/Compare Flags -- avoid accidental IRQ trigger */
        RC5_ClearInputTimeoutFlag();
        RC5_ClearInputCaptureFlag();

        /*Enable TIMx Input Capture interrupt*/
        RC5_EnableInputCaptureIRQ();

}


/**
 * \author        Dennis Hromin
 * \brief         RC5 Timer Base ISR. Always executed from lowlevel Main Timer ISR function part of project specific code.
 * \date          12/26/2008
 * \param         void
 * \return        void
 * \retval        none
**/
void IrReceiver::RC5_ProcessTimerISR (void)
{
/********************************************************************
 * Description  :This function is called when either the ir rising  *
 *                               or falling edge timer capture interrupt occurs.        *
 *   NOTE: Code ported from ILux Lighting device..
 *      Original Author: Hazrat Shaw
 ********************************************************************/

        //      Note: The interrupt execution time is estimated to be about 10usec (has not been verified yet!)

        UINT16  CaptureTime; //stores the 16-bit Timer Input Capture time
        UINT8 IRstate; //IRstate Flag to indicate whether to restart or not

/* Check the IRQ Status flag to determine what type of IRQ occurred... */
if( RC5_IsInputTimeoutIRQFlagSet() && RC5_IsInputTimeoutIRQEnabled() )
        {
        /* Always clear out the IRQ flag that was set before leaving this ISR */
        RC5_ClearInputTimeoutFlag();
        RC5_RestartCapture();
        }

else if (RC5_IsInputCaptureIRQFlagSet())
        {
        /*Disable TIMx Input Capture interrupt*/
        /*Disable TIMx Output Compare interrupt*/
        RC5_DisableInputCaptureIRQ();
        RC5_DisableInputTimeoutIRQ();

        /* Always clear out the IRQ flag that was set before leaving this ISR */
        RC5_ClearInputCaptureFlag();
        RC5_ClearInputTimeoutFlag();

        CaptureTime = RC5_GetCaptureTime();
        IRstate = g_IrReceiver->RC5_ProcessInputCapture(CaptureTime);

		/* If there was either a failure or we have completed, just set to Restart State */
        if( (IRstate == IR_DECODE_FAILURE) || (IRstate == IR_DECODE_COMPLETE) )
                {
                RC5_RestartCapture(); // Restart State for NEW Reception
                }
        /* Otherwise it was successful, so setup for next edge and continue decoding */
        else
                {
                //      Enable rising, or falling edge interrupt

                //      Set for falling or rising edge interrupt based on flag status
                if (m_RC5.NextSampleEdge)
                        RC5_SetInputCaptEdge_Rising();
                else
                        RC5_SetInputCaptEdge_Falling();

                //      receiving a packet, start packet-bit timeout
                /*Enable TIMx Output Compare (For RC5 Packet Overflow) interrupt*/
                RC5_SetTimeoutValue(CaptureTime);

                RC5_ClearInputCaptureFlag();
                RC5_ClearInputTimeoutFlag();

                /*Enable TIMx Input Capture interrupt*/
                RC5_EnableInputCaptureIRQ();
                RC5_EnableInputTimeoutIRQ();
                } //end else if it was success

        } //end else if for IC1 Flag Check

} //end ISR function



/**
 * \author        Dennis Hromin
 * \brief         Main RC5 Decoding ISR. Always executed from lowlevel Main Timer ISR function part of project specific code.
 * \date          12/26/2008
 * \param         void
 * \return        void
 * \retval        none
**/
BOOL IrReceiver::RC5_ProcessInputCapture(UINT16 CaptureTime)
{
/********************************************************************
 * Description  :This function is called when either the ir rising  *
 *                               or falling edge timer capture interrupt occurs.        *
 *   NOTE: Code ported from ILux Lighting device..
 *      Original Author: Hazrat Shaw
 ********************************************************************/

        BOOL    IRstate = IR_DECODE_SUCCESS;
        UINT16  i;

        /*-- If this is the 1st fallling puluse edge in packet ---*/
        if (m_RC5.SampleHalfBitsCnt == 0)
        {
                //      First falling edge
                m_RC5.SampleTime = CaptureTime; //update sample time with new captured time
                m_RC5.NextSampleEdge = 1;
                m_RC5.SampleHalfBitsCnt = 1;
        }
        /* -- Check if this is a start bit -- */
        else if (m_RC5.SampleHalfBitsCnt == 1)
        {
                //      Check for start bit
                if (((CaptureTime - m_RC5.SampleTime) > RC5_HALF_BIT_TIME_MIN) && ((CaptureTime - m_RC5.SampleTime) < RC5_HALF_BIT_TIME_MAX))
                {
                        m_RC5.SampleAcc = 2; //Stores 10 binary for the start bit
                        m_RC5.SampleTime = CaptureTime; //update sample time with new captured time
                        m_RC5.NextSampleEdge = 0;
                        m_RC5.SampleHalfBitsCnt++;
                }
                else
                {
                        IRstate = IR_DECODE_FAILURE;
                }
        }
        /*-- Otherwsie, this is another bit in the pulse stream after the start bit ---*/
        else
        {
                /*-- If new pulse is a HALF bit pulse, shift in a 0 or 1 based on edge ---*/
                if (((CaptureTime - m_RC5.SampleTime) > RC5_HALF_BIT_TIME_MIN) && ((CaptureTime - m_RC5.SampleTime) < RC5_HALF_BIT_TIME_MAX))
                {
                        m_RC5.SampleAcc <<= 1;
                        if (!m_RC5.NextSampleEdge)
                                m_RC5.SampleAcc |= 1;
                        m_RC5.SampleHalfBitsCnt++;
                }
                /*-- If new pulse is a FULL bit pulse, shift in 2 0's or 1's based on edge ---*/
                else if (((CaptureTime - m_RC5.SampleTime) > RC5_BIT_TIME_MIN) && ((CaptureTime - m_RC5.SampleTime) < RC5_BIT_TIME_MAX))
                {
                        m_RC5.SampleAcc <<= 2;
                        if (!m_RC5.NextSampleEdge)
                                m_RC5.SampleAcc |= 3;
                        m_RC5.SampleHalfBitsCnt+=2;
                }
                /*-- Otherwise pulse width was not in range.. discard packet ---*/
                else
                {
                        IRstate = IR_DECODE_FAILURE;
                }

                //      next expected edge (flip the state)
                m_RC5.NextSampleEdge ^= 1;
                //      Save sample time
                m_RC5.SampleTime = CaptureTime;

                if ((m_RC5.NextSampleEdge == 0) && (m_RC5.SampleHalfBitsCnt == ((RC5_MAX_BITS*2)-1)))
                {
                        //      The last bit is high, add to memory as received
                        m_RC5.SampleAcc <<= 1;
                        m_RC5.SampleAcc |= 1;
                        m_RC5.SampleHalfBitsCnt++;
                }
        }

        /*--- Checks if we recieved all required bits.. stores the final value ---*/
        if ((m_RC5.SampleHalfBitsCnt >= (RC5_MAX_BITS*2)) && (IRstate != IR_DECODE_FAILURE))
        {
                m_RC5.SampleAccImg = m_RC5.SampleAcc;
                //      full packet was received, convert received data to bit format packet
                for (i=0, m_RC5.TmpCode=0; i<RC5_MAX_BITS; i++)
                {
                        m_RC5.TmpCode >>= 1;
                        if ((m_RC5.SampleAcc & 0x03) == 0x2) {  m_RC5.TmpCode |= 0x8000;        }
                        m_RC5.SampleAcc >>= 2;
                }
                m_RC5.TmpCode >>= 2;
                //      check for valid start-bits
                if ((m_RC5.TmpCode & RC5_START_BIT_MASK_VER1) == RC5_START_BIT_MASK_VER1)
                {
                        //      New packet was received, submit it for processing
                        /*-- Add the data to the queue... ---*/
                        OsQueue(m_IrRC5Queue, (void *)&m_RC5.TmpCode, OS_NO_WAIT );
                        IRstate = IR_DECODE_COMPLETE;
                        m_RC5.IRCodeState = RC5_CODE_STATE_ACTIVE;
                }
                else
                {
                        //      error: invalid start pattern
                        IRstate = IR_DECODE_FAILURE;
                }

        }

        /*--- Final check if there was a failure, enqueue in a NULL RC5 code to force restart of Timeout TImer in Task Function ---*/
        if(IRstate == IR_DECODE_FAILURE)
                {
                if( m_RC5.IRCodeState == RC5_CODE_STATE_ACTIVE)
                        {
                        m_RC5.IRCodeState = RC5_CODE_STATE_RELEASE; //change states to "release" state
                        m_RC5.TmpCode = RC5_CODE_NULL;
                        //If we were previously in an "Active" IR Code State and processing a valid code, now we can safely send a release code due to error detection in input pulse stream
                        OsQueue(m_IrRC5Queue, (void *)&m_RC5.TmpCode, OS_NO_WAIT ); //queue up a release code of 0x00
                        }
                //Else, otherwise do nothing and just continue onward due to decode error detected
                }

return IRstate;  //Always return the restart status so base Timer ISR can parse what to do next
} //end decode function


/**
* \brief    Initializes/restarts IR input capture for pulses from a dock.
* \param    (None).
* \return   (None).
* \author   Dennis Hromin, Arto Kiremitdjian.
* \date     04/10/2012
**/
void IrReceiver::IrDock_RestartCapture(void)
{
    // Disable all associated interrupts.
    RC5_DisableInputTimeoutIRQ();
    RC5_DisableInputCaptureIRQ();

    // Reset key values to default start-up state.
    memset(&m_RC5, 0, sizeof(m_RC5)); // Reset/clear all state-machine data.
    m_RC5.NextSampleEdge = 1; // Always expect same kind of edge.

    // Configure next rising/falling edge interrupt.
    if (0 != m_RC5.NextSampleEdge)
    {
        RC5_SetInputCaptEdge_Rising();
    }
    else
    {
        RC5_SetInputCaptEdge_Falling();
    }
    // Clear all associated interrupts.
    RC5_ClearInputTimeoutFlag();
    RC5_ClearInputCaptureFlag();

    // Enable only the interrupts that are initially required.
    RC5_EnableInputCaptureIRQ();
}

#define MILLISECONDS_PER_SECOND (1000)
#define PULSE_PERIOD_MILLISECONDS (300)
// TODO: !! Cannot directly use symbols from "TSx_IRController_ir.h"....
//!!- #define PULSE_PERIOD_TICKS ((((IR_INPUT_CLK_MAIN * 2) / IR_INPUT_CLK_PRESCALER) * PULSE_PERIOD_MILLISECONDS) / MILLISECONDS_PER_SECOND)
#define PULSE_PERIOD_TICKS ((((36000000 * 2) / (1152 * 2)) * PULSE_PERIOD_MILLISECONDS) / MILLISECONDS_PER_SECOND)

/**
* \brief    Interrupt handler (ISR) for IR input capture or its time-out,
*           but only for detecting IR pulses from a dock (Not for RC-5).
* \detail   - Invoked by project-specific low-level main timer ISR function.
*           - Invoked upon rising/falling edge interrupt of IR input capture.
* \param    (None).
* \return   (None).
* \author   Dennis Griffin, Arto Kiremitdjian.
* \date     10/11/2011
**/
void IrReceiver::IrDock_ProcessTimerISR(void)
{
    // Determine what kinds of interrupts to process.
    bool const gotTimeout = (RC5_IsInputTimeoutIRQFlagSet() &&
                             RC5_IsInputTimeoutIRQEnabled());
    bool const gotCapture = (RC5_IsInputCaptureIRQFlagSet() &&
                             RC5_IsInputCaptureIRQEnabled());

    UINT16 const captureTime = RC5_GetCaptureTime();

    // ISR is now running: Disable and clear all associated interrupts.
    RC5_DisableInputTimeoutIRQ();
    RC5_DisableInputCaptureIRQ();
    RC5_ClearInputTimeoutFlag();
    RC5_ClearInputCaptureFlag();

    // Handle pending interrupts, in order of priority.
    if (gotTimeout && (5 > m_RC5.SampleHalfBitsCnt) && (0 < m_RC5.SampleAcc))
    {
        // Successful end of pulse-detection time period 1..4.
        // Prepare for next time period.
        m_RC5.SampleAcc = 0; // Reset/clear counter.
        ++m_RC5.SampleHalfBitsCnt; // Advance to next state.
        RC5_SetTimeoutValue(captureTime + PULSE_PERIOD_TICKS); // Next time-out.

        // Enable any interrupts expected after this ISR.
        RC5_EnableInputTimeoutIRQ();
        RC5_EnableInputCaptureIRQ();
    }
    else if (gotCapture &&
             g_IrReceiver->ProcessDockInputCapture(captureTime))
    {
        // Receiving an IR-pulse stream.
        // Callee (above) is responsible for enabling any interrupts.
    }
    else
    {
        // Failed to receive IR-pulse stream, Retry.
        IrDock_RestartCapture();
    }
}


/**
* \brief    Decodes next part of IR input stream when an IR rising/falling
*           edge timer capture interrupt occurs (Invoked by an ISR).
* \param    CaptureTime - IR input capture timer value, in ticks.
* \return   Boolean: Whether or not this function succeeded.
* \author   Arto Kiremitdjian.
* \date     4/13/2012.
**/
BOOL IrReceiver::ProcessDockInputCapture(UINT16 CaptureTime)
{
    BOOL result = TRUE; // ASSUME success.

    switch (m_RC5.SampleHalfBitsCnt)
    {
    case 0: // Completed "infinite" wait for first captured edge.
        ++m_RC5.SampleHalfBitsCnt; // Advance to next state.

        // Prepare to track edges for a fixed time period, using a time-out.
        RC5_SetTimeoutValue(CaptureTime + PULSE_PERIOD_TICKS);

        // Enable any interrupts expected after current ISR.
        RC5_EnableInputTimeoutIRQ();
        RC5_EnableInputCaptureIRQ();
        break;

    case 1: case 2: case 3: case 4: // In a typical pulse-detection period.
        //- - - - -
        // Do not change current state upon input-capture:
        //  State changes only upon time-out (outside this function).
        //- - - - -
        ++m_RC5.SampleAcc; // Count edges during the current period.

        // Enable any interrupts expected after current ISR.
        RC5_EnableInputTimeoutIRQ();
        RC5_EnableInputCaptureIRQ();
        break;

    case 5: // Early finish of the decisive pulse-detection period.
        //- - - - -
        // Since this is the decisive pulse-detection period, a single pulse
        //  indicates final success.  There is no need for future interrupts.
        //- - - - -
        ++m_RC5.SampleHalfBitsCnt; // Advance to next state.

        // Now assumes that the IR pulses are from a dock.
        //
        //  - NOTE: The value below represents a special, internal, simulated
        //          RC-5 code, instead of a code actually received over IR.
        //          (The TST-600 main CPU receives Cresnet packets over I2C
        //          from this STM32 MCU, and it should expect the special IR
        //          code as internal inidication that an IR dock is in use.)
        //
        m_RC5.TmpCode = 0x060A; // Becomes 0x78 0x30 0x12 in Cresnet "1-Way RF".
        OsQueue(m_IrRC5Queue, &m_RC5.TmpCode, OS_NO_WAIT);
        break;

    case 6:
        // Nothing to do: Remain in current state forever.
        break;

    default: // UNEXPECTED CONDITION.
        result = FALSE; // Prepare to report error/failure.
        break;
    }
    return result;
}
